import matplotlib.pyplot as plt
import numpy as np
import time
from scipy.optimize import minimize_scalar

plt.rcParams['pdf.fonttype'] = 42
plt.rcParams['ps.fonttype'] = 42

#####################################################################################
#############################
#############################     This code runs RM+, Predictive RM+, Smooth PRM+, ExRM+,
#############################     Extragradient algorithm and Predictive Online Mirror Descent 
#############################     all with/without alternation, and different weighting schemes.
############################# 
#############################   Line 23 -> Line 1300: define auxiliary functions and all the algorithms.
#############################   The rest of the code is the simulation environment.
############################# 
############################# 
############################# 
############################# 
#####################################################################################

######## Auxiliary functions

def compute_next_decision(aggregate_payoff):
    
    '''
    Compute next decision based on Blackwell approachability
    Input: 
        projected) aggregate_payoff, vector in R_+^p
    Output: 
        next decision
    '''
    
    if max(aggregate_payoff) == 0:
        return (1/(len(aggregate_payoff)))*np.ones(len(aggregate_payoff))
    else:
        return aggregate_payoff/sum(aggregate_payoff)

def compute_next_aggregate_payoff(aggregate_payoff,x,c_x):
    
    '''
    Compute next aggregate payoff
    Input: 
        aggregate_payoff: vector in R_+^p
        x: last decision
        c_x: gradient from x
    
    Output: 
        next aggregate payoff
    '''
    
    return aggregate_payoff + np.dot(c_x,x)*np.ones(len(x))-c_x

def projection_onto_nonnegative_orthant(aggregate_payoff):
    
    '''
    Compute thresholded aggregate payoff
    Input: 
        aggregate_payoff, vector in R^p
    Output: 
        projection of aggregate_payoff onto R_+^p
    '''
    
    return np.maximum(aggregate_payoff,np.zeros(len(aggregate_payoff)))  

def projection_onto_chopped_orthant(y):
    
    '''
    Compute the projection onto {R | R >=0 , <R,1> >= 1}
    Input:
        y vector of dimension p
    Output:
        orthogonal projection of y onto {R | R >=0 , <R,1> >= 1}
    '''
    
    y_loc = np.maximum(y,0)
    if sum(y_loc) >= 1:
        return y_loc
    else:
        return euclidean_proj_simplex(y)

def fixed_point_iteration(R_x,R_y,k,A,eta):
    
    '''
    
    Compute k iterations of fixed-point iteration
    R_x,R_y: current aggregate payoffs of both players
    x0,y0: initial point for the fixed-point iterations
    k: number of iterations
    A: matrix of payoffs
    
    Return:
        w_x,w_y: last-but-one point in the fixed-point iteration
        z_x,z_y: last point in the fixed-point iteration
        
    Note: k=2 corresponds to ExRM+
    '''
    
    A_T = np.transpose(A)
    
    w_x = R_x
    w_y = R_y
    
    x = compute_next_decision(projection_onto_chopped_orthant(w_x))
    y = compute_next_decision(projection_onto_chopped_orthant(w_y))
    
    # print('New fixed-point computation...')
    
    for iter in range(k-1):
        
        # Compute next losses
        c_x = eta*np.dot(A,y)
        c_y = - eta*np.dot(A_T,x)
        
        # Update aggregate payoffs
        R_loc_x=compute_next_aggregate_payoff(R_x,x,c_x)
        R_loc_y=compute_next_aggregate_payoff(R_y,y,c_y)
            
        # Compute next decision for x-player
        w_x = projection_onto_chopped_orthant(R_loc_x)
        x=compute_next_decision(w_x)
        
        # Compute next decision for y-player
        w_y = projection_onto_chopped_orthant(R_loc_y)
        y=compute_next_decision(w_y)

    # Compute next losses
    c_x = eta*np.dot(A,y)
    c_y = - eta*np.dot(A_T,x)
    
    # Update aggregate payoffs
    R_loc_x=compute_next_aggregate_payoff(R_x,x,c_x)
    R_loc_y=compute_next_aggregate_payoff(R_y,y,c_y)
        
    # jgc: this works for RM+ but this would not work for RM.
    # Compute next decision for x-player
    z_x = projection_onto_chopped_orthant(R_loc_x)
    
    # Compute next decision for y-player
    z_y = projection_onto_chopped_orthant(R_loc_y)
        
    return w_x,w_y,z_x,z_y

def euclidean_proj_simplex(v):
    
    '''
    Compute the orthogonal projection onto the simplex
    from https://gist.github.com/EdwardRaff/f4f4cf0c927c2addfb39
    '''
    
    n_features = v.shape[0]
    u = np.sort(v)[::-1]
    # print('u shape', u.shape)
    cssv = np.cumsum(u) - 1
    # print('cssv shape', cssv.shape)
    ind = np.arange(n_features) + 1
    # print('ind shape', ind.shape)
    cond = u - cssv / ind > 0
    rho = ind[cond][-1]
    theta = cssv[cond][-1] / float(rho)
    # print('opt mu:',np.around(theta,1))
    w = np.maximum(v - theta, 0)
    
    return w
      
##########################
#########
#########     (Predictive ) RM+ with/without alternation
#########
##########################

def regret_matching_plus(T,A,n,m,q,R0):    
    
    '''
    
    Regret Matching Plus (without alternation)
    
    Algorithm 1 in the submission: "Last-Iterate Convergence Properties of Regret-Matching Algorithms in Games."
    
    Input:
        T = number of iterations
        A = payoff matrix
        n,m = dimensions of simplices of each player
        q = exponent for the weights t**q
        R0 = initialization point
        
    Output:
        5 Arrays of dimension T:
            result_duality_gap: duality gap of current iterates 
            results_x: x[t] at each iteration t=0,...T-1
            results_y: y[t] at each iteration t=0,...T-1
            results_regret_x: regret accumulated at each iteration for the x-player
            results_regret_y: regret accumulated at each iteration for the y-player
        
    '''
    
    A_T=np.transpose(A)
    
    result_duality_gap = np.zeros(T-1)
    results_regret_x = np.zeros(T-1)
    results_regret_y = np.zeros(T-1)
    results_x = np.zeros((T-1,n))
    results_y = np.zeros((T-1,m))
    
    sum_instant_regret_x = 0
    sum_instant_regret_y = 0    
    
    # initialization
    x_tot=np.zeros(n)
    y_tot=np.zeros(m)
    
    # running sum of iterates with uniform weights
    x_tot_unf=np.zeros(n)
    y_tot_unf=np.zeros(m)

    w_tot = 0
    
    R_x=np.maximum(np.zeros(n),R0*np.ones(n)) 
    R_y=np.maximum(np.zeros(m),R0*np.ones(m)) 
    
    for t in range(1,T):
        
        # current weights
        w_t = t**q
        
        # Compute next decision for x-player
        R_x = projection_onto_nonnegative_orthant(R_x)
        x=compute_next_decision(R_x)
        
        # Compute next decision for y-player
        R_y = projection_onto_nonnegative_orthant(R_y)
        y=compute_next_decision(R_y)

        # Update aggregate payoff for y-player
        c_y = - np.dot(A_T,x)
        R_y=compute_next_aggregate_payoff(R_y,y,c_y)
        
        sum_instant_regret_y += np.dot(y,c_y)

        # Update aggregate payoff for x-player
        c_x = np.dot(A,y)
        R_x=compute_next_aggregate_payoff(R_x,x,c_x)
        
        sum_instant_regret_x += np.dot(x,c_x)
        
        # Update running sums of decisions
        x_tot += w_t*x
        y_tot += w_t*y
        w_tot += w_t
        
        x_tot_unf += x
        y_tot_unf += y
        
        # Compute duality gap for current iterates
        result_duality_gap[t-1]=np.max(np.dot(A_T,x))-np.min(np.dot(A,y))
        results_x[t-1]=x
        results_y[t-1]=y
        results_regret_x[t-1] = sum_instant_regret_x-np.min(np.dot(A,y_tot_unf))
        results_regret_y[t-1] = np.max(np.dot(A_T,x_tot_unf))+sum_instant_regret_y
        
    return result_duality_gap, results_x, results_y, results_regret_x, results_regret_y

def regret_matching_alt_plus(T,A,n,m,q,R0):    
    
    '''
    
    Regret Matching Plus (with alternation)
    
    Algorithm 7 in the submission: "Last-Iterate Convergence Properties of Regret-Matching Algorithms in Games."


    Input:
        T = number of iterations
        A = payoff matrix
        n,m = dimensions of simplices of each player
        q = exponent for the weights t**q
        R0 = initialization point
        
    Output:
        5 Arrays of dimension T:
            result_duality_gap: duality gap of current iterates 
            results_x: x[t] at each iteration t=0,...T-1
            results_y: y[t] at each iteration t=0,...T-1
            results_regret_x: regret accumulated at each iteration for the x-player
            results_regret_y: regret accumulated at each iteration for the y-player
        
    '''
    
    A_T=np.transpose(A)
    
    result_duality_gap = np.zeros(T-1)
    results_regret_x = np.zeros(T-1)
    results_regret_y = np.zeros(T-1)
    results_x = np.zeros((T-1,n))
    results_y = np.zeros((T-1,m))
    
    sum_instant_regret_x = 0
    sum_instant_regret_y = 0    
    
    # initialization
    x_tot=np.zeros(n)
    y_tot=np.zeros(m)
    w_tot = 0
    
    
    # running sum of iterates with uniform weights
    x_tot_unf=np.zeros(n)
    y_tot_unf=np.zeros(m)
    
    R_x=np.maximum(np.zeros(n),R0*np.ones(n)) 
    R_y=np.maximum(np.zeros(m),R0*np.ones(m)) 
    
    # First decision for y
    y = (1/m)*np.ones(m)
    
    # local loss vector for the x-player
    c_x = np.dot(A,y)
    
    for t in range(1,T):
        
        w_t = t**q
        
        # Compute next decision for x-player
        R_x = projection_onto_nonnegative_orthant(R_x)
        x=compute_next_decision(R_x)

        # Update aggregate payoff for y-player
        c_y = - np.dot(A_T,x)
        R_y=compute_next_aggregate_payoff(R_y,y,c_y)
        
        # Compute next decision for y-player
        R_y = projection_onto_nonnegative_orthant(R_y)
                
        y=compute_next_decision(R_y)
        
        sum_instant_regret_y += np.dot(y,c_y)

        
        # Compute next loss for x-player
        c_x = np.dot(A,y)
        sum_instant_regret_x += np.dot(x,c_x)
        
        # Update aggregate payoff for x-player
        R_x=compute_next_aggregate_payoff(R_x,x,c_x)
        
        # Update running sums of decisions
        y_tot += w_t*y
        x_tot += w_t*x
        
        x_tot_unf += x
        y_tot_unf += y
        
        w_tot += w_t
        
        # Compute duality gap for current iterates
        result_duality_gap[t-1]=np.max(np.dot(A_T,x))-np.min(np.dot(A,y))
        results_x[t-1]=x
        results_y[t-1]=y
        results_regret_x[t-1] = sum_instant_regret_x-np.min(np.dot(A,y_tot_unf))
        results_regret_y[t-1] = np.max(np.dot(A_T,x_tot_unf))+sum_instant_regret_y
        
    return result_duality_gap, results_x, results_y, results_regret_x, results_regret_y

def optimistic_regret_matching_plus(T,A,n,m,q,R0):    
    
    '''
    
    Optimistic Regret Matching Plus (without alternation)
    
    Algorithm 2 in the submission: "Last-Iterate Convergence Properties of Regret-Matching Algorithms in Games."


    Input:
        T = number of iterations
        A = payoff matrix
        n,m = dimensions of simplices of each player
        q = exponent for the weights t**q
        R0 = initialization point
        
    Output:
        5 Arrays of dimension T:
            result_duality_gap: duality gap of current iterates
            results_x: x[t] at each iteration t=0,...T-1
            results_y: y[t] at each iteration t=0,...T-1
            results_regret_x: regret accumulated at each iteration for the x-player
            results_regret_y: regret accumulated at each iteration for the y-player
        
    '''
    
    A_T=np.transpose(A)
    
    result_duality_gap = np.zeros(T-1)
    results_regret_x = np.zeros(T-1)
    results_regret_y = np.zeros(T-1)
    results_x = np.zeros((T-1,n))
    results_y = np.zeros((T-1,m))
    
    # initialization
    x_tot=np.zeros(n)
    y_tot=np.zeros(m)    
    
    # running sum of iterates with uniform weights
    x_tot_unf=np.zeros(n)
    y_tot_unf=np.zeros(m)
    
    sum_instant_regret_x = 0
    sum_instant_regret_y = 0  
    
    z_R_x=np.maximum(np.zeros(n),R0*np.ones(n)) 
    z_R_y=np.maximum(np.zeros(m),R0*np.ones(m)) 
    
    z_R_x = (1/sum(z_R_x))*z_R_x
    z_R_y = (1/sum(z_R_y))*z_R_y
    
    w_R_x = z_R_x.copy()
    w_R_y = z_R_y.copy()
    
    x_loc = compute_next_decision(z_R_x)
    y_loc = compute_next_decision(z_R_y)
    
    c_x = np.dot(A,y_loc)
    c_y = - np.dot(A_T,x_loc)
    
    for t in range(1,T):
        
        # z_R : z^{t-1}
        # w_R : w^t

        z_R_x = projection_onto_nonnegative_orthant(compute_next_aggregate_payoff(w_R_x,x_loc,c_x))
        x_loc=compute_next_decision(z_R_x)
        
        # Compute next decision for y-player
        z_R_y = projection_onto_nonnegative_orthant(compute_next_aggregate_payoff(w_R_y,y_loc,c_y))
        y_loc=compute_next_decision(z_R_y)
        
        c_x = np.dot(A,y_loc)
        c_y = - np.dot(A_T,x_loc)
        
        # Update aggregate payoffs
        w_R_x=projection_onto_nonnegative_orthant(compute_next_aggregate_payoff(w_R_x,x_loc,c_x))
        w_R_y=projection_onto_nonnegative_orthant(compute_next_aggregate_payoff(w_R_y,y_loc,c_y))
        
        x=compute_next_decision(w_R_x)
        y=compute_next_decision(w_R_y)
        
        sum_instant_regret_x += np.dot(x,c_x)
        sum_instant_regret_y += np.dot(y,c_y)
        
        # Update running sums of decisions
        x_tot += x
        y_tot += y
        
        x_tot_unf += x
        y_tot_unf += y
            
        # Compute duality gap for current iterates
        result_duality_gap[t-1]=np.max(np.dot(A_T,x))-np.min(np.dot(A,y))
        results_x[t-1]=x
        results_y[t-1]=y
        results_regret_x[t-1] = sum_instant_regret_x-np.min(np.dot(A,y_tot_unf))
        results_regret_y[t-1] = np.max(np.dot(A_T,x_tot_unf))+sum_instant_regret_y
        
    return result_duality_gap, results_x, results_y, results_regret_x, results_regret_y

def optimistic_regret_matching_plus_alt(T,A,n,m,q,R0):    
    
    '''
    
    Optimistic Regret Matching Plus (with alternation)
    
    Algorithm 8 in the submission: "Last-Iterate Convergence Properties of Regret-Matching Algorithms in Games."


    Input:
        T = number of iterations
        A = payoff matrix
        n,m = dimensions of simplices of each player
        q = exponent for the weights t**q
        R0 = initialization point
        
    Output:
        5 Arrays of dimension T:
            result_duality_gap: duality gap of current iterates
            results_x: x[t] at each iteration t=0,...T-1
            results_y: y[t] at each iteration t=0,...T-1
            results_regret_x: regret accumulated at each iteration for the x-player
            results_regret_y: regret accumulated at each iteration for the y-player
        
    '''
    
    A_T=np.transpose(A)
    
    result_duality_gap = np.zeros(T-1)
    results_regret_x = np.zeros(T-1)
    results_regret_y = np.zeros(T-1)
    results_x = np.zeros((T-1,n))
    results_y = np.zeros((T-1,m))
    
    # initialization
    x_tot=np.zeros(n)
    y_tot=np.zeros(m)    
    
    # running sum of iterates with uniform weights
    x_tot_unf=np.zeros(n)
    y_tot_unf=np.zeros(m)
    
    sum_instant_regret_x = 0
    sum_instant_regret_y = 0  
    
    z_R_x=np.maximum(np.zeros(n),R0*np.ones(n)) 
    z_R_y=np.maximum(np.zeros(m),R0*np.ones(m)) 
    
    z_R_x = (1/sum(z_R_x))*z_R_x
    z_R_y = (1/sum(z_R_y))*z_R_y
    
    w_R_x = z_R_x.copy()
    w_R_y = z_R_y.copy()
    
    x_loc = compute_next_decision(z_R_x)
    y_loc = compute_next_decision(z_R_y)
    
    c_x = np.dot(A,y_loc)
    c_y = - np.dot(A_T,x_loc)
    
    for t in range(1,T):
        
        # z_R : z^{t-1}
        # w_R : w^t

        z_R_x = projection_onto_nonnegative_orthant(compute_next_aggregate_payoff(w_R_x,x_loc,c_x))
        x_loc=compute_next_decision(z_R_x)
        
        
        c_y = - np.dot(A_T,x_loc)
        w_R_y=projection_onto_nonnegative_orthant(compute_next_aggregate_payoff(w_R_y,y_loc,c_y))
        
        # Compute next decision for y-player
        z_R_y = projection_onto_nonnegative_orthant(compute_next_aggregate_payoff(w_R_y,y_loc,c_y))
        y_loc=compute_next_decision(z_R_y)
        y=compute_next_decision(w_R_y)

        c_x = np.dot(A,y_loc)
        
        # Update aggregate payoffs
        w_R_x=projection_onto_nonnegative_orthant(compute_next_aggregate_payoff(w_R_x,x_loc,c_x))
        
        x=compute_next_decision(w_R_x)
        
        sum_instant_regret_x += np.dot(x,c_x)
        sum_instant_regret_y += np.dot(y,c_y)
        
        # Update running sums of decisions
        x_tot += x
        y_tot += y
        
        x_tot_unf += x
        y_tot_unf += y
            
        # Compute duality gap for current iterates
        result_duality_gap[t-1]=np.max(np.dot(A_T,x))-np.min(np.dot(A,y))
        results_x[t-1]=x
        results_y[t-1]=y
        results_regret_x[t-1] = sum_instant_regret_x-np.min(np.dot(A,y_tot_unf))
        results_regret_y[t-1] = np.max(np.dot(A_T,x_tot_unf))+sum_instant_regret_y
        
    return result_duality_gap, results_x, results_y, results_regret_x, results_regret_y

##########################
#########
#########     Algorithms with chopping off and restarting: Conceptual RM+ (chopping off),
#########     Smooth PRM+ (chopping off), Stable PRM+ (restarting)
#########
##########################

def conceptual_regret_matching_plus_chopped(T,A,n,m,q,k,R0,eta):
    
    '''
    
    Extragradient RM+ (with smoothing)
    
    Algorithm 3 in the submission: "Last-Iterate Convergence Properties of Regret-Matching Algorithms in Games."


    Input:
        T = number of iterations
        A = payoff matrix
        n,m = dimensions of simplices of each player
        q = exponent for the weights t**q
        k = number of fixed point iterations; k=2: "Mirror Prox" type of algorithm
        R0 = initialization point
        eta = step size
        
    Output:
        5 Arrays of dimension T:
            result_duality_gap: duality gap of current iterates
            results_x: x[t] at each iteration t=0,...T-1
            results_y: y[t] at each iteration t=0,...T-1
            results_regret_x: regret accumulated at each iteration for the x-player
            results_regret_y: regret accumulated at each iteration for the y-player
        
    '''
    
    A_T=np.transpose(A)
    
    result_duality_gap = np.zeros(T-1)
    results_regret_x = np.zeros(T-1)
    results_regret_y = np.zeros(T-1)
    results_x = np.zeros((T-1,n))
    results_y = np.zeros((T-1,m))
    
    sum_instant_regret_x = 0
    sum_instant_regret_y = 0  
    
    # initialization
    x_tot=np.zeros(n)
    y_tot=np.zeros(m)
    w_tot = 0    
    
    
    # running sum of iterates with uniform weights
    x_tot_unf=np.zeros(n)
    y_tot_unf=np.zeros(m)
    

    R_x=np.maximum(np.zeros(n),R0*np.ones(n)) 
    R_y=np.maximum(np.zeros(m),R0*np.ones(m)) 
    

    R_x = (1/sum(R_x))*R_x
    R_y = (1/sum(R_y))*R_y
    
    # initial decisions
    x = compute_next_decision(projection_onto_chopped_orthant(R_x))
    y = compute_next_decision(projection_onto_chopped_orthant(R_y))
    
    for t in range(1,T):
        
        # run fixed-point iteration for k iterations
        # intermediate point: w_x,w_y
        # end point: R_x, R_y
        w_x,w_y,R_x,R_y = fixed_point_iteration(R_x,R_y,k,A,eta)

        # Compute next decision for x-player
        x=compute_next_decision(w_x)
        
        # Compute next decision for y-player
        y=compute_next_decision(w_y)
        
        c_x = np.dot(A,y)
        c_y = - np.dot(A_T,x)
        
        sum_instant_regret_x += np.dot(x,c_x)
        sum_instant_regret_y += np.dot(y,c_y)
        
        # Update running sums of decisions
        
        w_t = (t**q)
        x_tot += w_t*x
        y_tot += w_t*y
        w_tot += w_t
        
        x_tot_unf += x
        y_tot_unf += y


        # Compute duality gap for current iterates
        result_duality_gap[t-1]=np.max(np.dot(A_T,x))-np.min(np.dot(A,y))
        results_x[t-1]=x
        results_y[t-1]=y
        results_regret_x[t-1] = sum_instant_regret_x-np.min(np.dot(A,y_tot_unf))
        results_regret_y[t-1] = np.max(np.dot(A_T,x_tot_unf))+sum_instant_regret_y
        
    return result_duality_gap, results_x, results_y, results_regret_x, results_regret_y


def optimistic_regret_matching_plus_chopped(T,A,n,m,q,R0,eta):    
    
    '''
    
    Smooth Predictive Regret Matching+ (SPRM+)
    
    Algorithm 4 in the submission: "Last-Iterate Convergence Properties of Regret-Matching Algorithms in Games."

    
    Input:
        T = number of iterations
        A = payoff matrix
        n,m = dimensions of simplices of each player
        q = exponent for the weights t**q
        R0 = initialization point
        
    Output:
        5 Arrays of dimension T:
            result_duality_gap: duality gap of current iterates
            results_x: x[t] at each iteration t=0,...T-1
            results_y: y[t] at each iteration t=0,...T-1
            results_regret_x: regret accumulated at each iteration for the x-player
            results_regret_y: regret accumulated at each iteration for the y-player
        
    '''
    
    A_T=np.transpose(A)
    
    result_duality_gap = np.zeros(T-1)
    results_regret_x = np.zeros(T-1)
    results_regret_y = np.zeros(T-1)
    results_x = np.zeros((T-1,n))
    results_y = np.zeros((T-1,m))
    
    # initialization
    x_tot=np.zeros(n)
    y_tot=np.zeros(m)    
    
    # running sum of iterates with uniform weights
    x_tot_unf=np.zeros(n)
    y_tot_unf=np.zeros(m)
    
    sum_instant_regret_x = 0
    sum_instant_regret_y = 0  
    
    z_R_x=np.maximum(np.zeros(n),R0*np.ones(n)) 
    z_R_y=np.maximum(np.zeros(m),R0*np.ones(m)) 
    
    z_R_x = (1/sum(z_R_x))*z_R_x
    z_R_y = (1/sum(z_R_y))*z_R_y
    
    w_R_x = z_R_x.copy()
    w_R_y = z_R_y.copy()
    
    x_loc = compute_next_decision(z_R_x)
    y_loc = compute_next_decision(z_R_y)
    
    c_x = eta*np.dot(A,y_loc)
    c_y = - eta*np.dot(A_T,x_loc)
    
    for t in range(1,T):
        
        # z_R : z^{t-1}
        # w_R : w^t

        z_R_x = projection_onto_chopped_orthant(compute_next_aggregate_payoff(w_R_x,x_loc,c_x))
        x_loc=compute_next_decision(z_R_x)
        
        # Compute next decision for y-player
        z_R_y = projection_onto_chopped_orthant(compute_next_aggregate_payoff(w_R_y,y_loc,c_y))
        y_loc=compute_next_decision(z_R_y)
        
        c_x = eta*np.dot(A,y_loc)
        c_y = - eta*np.dot(A_T,x_loc)
        
        # Update aggregate payoffs
        w_R_x=projection_onto_chopped_orthant(compute_next_aggregate_payoff(w_R_x,x_loc,c_x))
        w_R_y=projection_onto_chopped_orthant(compute_next_aggregate_payoff(w_R_y,y_loc,c_y))
        
        x=compute_next_decision(w_R_x)
        y=compute_next_decision(w_R_y)
        
        sum_instant_regret_x += np.dot(x,c_x)
        sum_instant_regret_y += np.dot(y,c_y)
        
        # Update running sums of decisions
        x_tot += x
        y_tot += y
        
        x_tot_unf += x
        y_tot_unf += y
            
        # Compute duality gap for current iterates
        result_duality_gap[t-1]=np.max(np.dot(A_T,x))-np.min(np.dot(A,y))
        results_x[t-1]=x
        results_y[t-1]=y
        results_regret_x[t-1] = sum_instant_regret_x-np.min(np.dot(A,y_tot_unf))
        results_regret_y[t-1] = np.max(np.dot(A_T,x_tot_unf))+sum_instant_regret_y
        
    return result_duality_gap, results_x, results_y, results_regret_x, results_regret_y

def optimistic_regret_matching_plus_alt_chopped(T,A,n,m,q,R0,eta):    
    
    '''
    
    Smooth Predictive Regret Matching+ (with alternation)
    Input:
        T = number of iterations
        A = payoff matrix
        n,m = dimensions of simplices of each player
        q = exponent for the weights t**q
        R0 = initialization point
        
    Output:
        5 Arrays of dimension T:
            result_duality_gap: duality gap of current iterates
            results_x: x[t] at each iteration t=0,...T-1
            results_y: y[t] at each iteration t=0,...T-1
            results_regret_x: regret accumulated at each iteration for the x-player
            results_regret_y: regret accumulated at each iteration for the y-player
        
    '''
    
    A_T=np.transpose(A)
    
    result_duality_gap = np.zeros(T-1)
    results_regret_x = np.zeros(T-1)
    results_regret_y = np.zeros(T-1)
    results_x = np.zeros((T-1,n))
    results_y = np.zeros((T-1,m))
    
    # initialization
    x_tot=np.zeros(n)
    y_tot=np.zeros(m)    
    
    # running sum of iterates with uniform weights
    x_tot_unf=np.zeros(n)
    y_tot_unf=np.zeros(m)
    
    sum_instant_regret_x = 0
    sum_instant_regret_y = 0  
    
    z_R_x=np.maximum(np.zeros(n),R0*np.ones(n)) 
    z_R_y=np.maximum(np.zeros(m),R0*np.ones(m)) 
    
    z_R_x = (1/sum(z_R_x))*z_R_x
    z_R_y = (1/sum(z_R_y))*z_R_y
    
    w_R_x = z_R_x.copy()
    w_R_y = z_R_y.copy()
    
    x_loc = compute_next_decision(z_R_x)
    y_loc = compute_next_decision(z_R_y)
    
    c_x = eta*np.dot(A,y_loc)
    c_y = - eta*np.dot(A_T,x_loc)
    
    for t in range(1,T):
        
        # z_R : z^{t-1}
        # w_R : w^t

        z_R_x = projection_onto_chopped_orthant(compute_next_aggregate_payoff(w_R_x,x_loc,c_x))
        x_loc=compute_next_decision(z_R_x)
        
        
        c_y = - eta*np.dot(A_T,x_loc)
        w_R_y=projection_onto_chopped_orthant(compute_next_aggregate_payoff(w_R_y,y_loc,c_y))
        
        # Compute next decision for y-player
        z_R_y = projection_onto_chopped_orthant(compute_next_aggregate_payoff(w_R_y,y_loc,c_y))
        y_loc=compute_next_decision(z_R_y)
        y=compute_next_decision(w_R_y)

        c_x = eta*np.dot(A,y_loc)
        
        # Update aggregate payoffs
        w_R_x=projection_onto_chopped_orthant(compute_next_aggregate_payoff(w_R_x,x_loc,c_x))
        
        x=compute_next_decision(w_R_x)
        
        sum_instant_regret_x += np.dot(x,c_x)
        sum_instant_regret_y += np.dot(y,c_y)
        
        # Update running sums of decisions
        x_tot += x
        y_tot += y
        
        x_tot_unf += x
        y_tot_unf += y
            
        # Compute duality gap for current iterates
        result_duality_gap[t-1]=np.max(np.dot(A_T,x))-np.min(np.dot(A,y))
        results_x[t-1]=x
        results_y[t-1]=y
        results_regret_x[t-1] = sum_instant_regret_x-np.min(np.dot(A,y_tot_unf))
        results_regret_y[t-1] = np.max(np.dot(A_T,x_tot_unf))+sum_instant_regret_y
        
    return result_duality_gap, results_x, results_y, results_regret_x, results_regret_y

def conceptual_regret_matching_plus_chopped_restarting(T,A,n,m,q,k,R0,eta):
    
    '''
    
    ExRM+ with restarting 
    Algorithm 5 in the submission: "Last-Iterate Convergence Properties of Regret-Matching Algorithms in Games."
    Input:
        T = number of iterations
        A = payoff matrix
        n,m = dimensions of simplices of each player
        q = exponent for the weights t**q
        k = number of fixed point iterations; k=2: "Mirror Prox" type of algorithm
        R0 = initialization point
        eta = step size
        
    Output:
        5 Arrays of dimension T:
            result_duality_gap: duality gap of current iterates
            results_x: x[t] at each iteration t=0,...T-1
            results_y: y[t] at each iteration t=0,...T-1
            results_regret_x: regret accumulated at each iteration for the x-player
            results_regret_y: regret accumulated at each iteration for the y-player
        
    '''
    
    A_T=np.transpose(A)
    
    result_duality_gap = np.zeros(T-1)
    results_regret_x = np.zeros(T-1)
    results_regret_y = np.zeros(T-1)
    results_x = np.zeros((T-1,n))
    results_y = np.zeros((T-1,m))
    
    sum_instant_regret_x = 0
    sum_instant_regret_y = 0  
    
    # initialization
    x_tot=np.zeros(n)
    y_tot=np.zeros(m)
    w_tot = 0    
    
    
    # running sum of iterates with uniform weights
    x_tot_unf=np.zeros(n)
    y_tot_unf=np.zeros(m)
    

    R_x=np.maximum(np.zeros(n),R0*np.ones(n)) 
    R_y=np.maximum(np.zeros(m),R0*np.ones(m)) 
    
    R_x = (1/sum(R_x))*R_x
    R_y = (1/sum(R_y))*R_y
    
    # initial decisions
    x = compute_next_decision(projection_onto_chopped_orthant(R_x))
    y = compute_next_decision(projection_onto_chopped_orthant(R_y))
    
    k_restart = 0
    
    for t in range(1,T):
        
        # run fixed-point iteration for k iterations
        # intermediate point: w_x,w_y
        # end point: R_x, R_y
        w_x,w_y,R_x,R_y = fixed_point_iteration(R_x,R_y,k,A,eta)

        # Compute next decision for x-player
        x=compute_next_decision(w_x)
        
        # Compute next decision for y-player
        y=compute_next_decision(w_y)
        
        c_x = np.dot(A,y)
        c_y = - np.dot(A_T,x)
        
        sum_instant_regret_x += np.dot(x,c_x)
        sum_instant_regret_y += np.dot(y,c_y)
        
        # Update running sums of decisions
        
        w_t = (t**q)
        x_tot += w_t*x
        y_tot += w_t*y
        w_tot += w_t
        
        x_tot_unf += x
        y_tot_unf += y
        
        # checking if we need to restart:
        if np.sqrt(np.linalg.norm(w_x-R_x)**2 + np.linalg.norm(w_y-R_y)**2)< 1/(2**k_restart):
            k_restart +=1
            R_x = x
            R_y = y

        # Compute duality gap for current iterates
        result_duality_gap[t-1]=np.max(np.dot(A_T,x))-np.min(np.dot(A,y))
        results_x[t-1]=x
        results_y[t-1]=y
        results_regret_x[t-1] = sum_instant_regret_x-np.min(np.dot(A,y_tot_unf))
        results_regret_y[t-1] = np.max(np.dot(A_T,x_tot_unf))+sum_instant_regret_y
        
        
        
    return result_duality_gap, results_x, results_y, results_regret_x, results_regret_y

def optimistic_regret_matching_plus_chopped_restarting(T,A,n,m,q,R0,eta):    
    
    '''
    
    Optimistic Regret Matching Plus with restarting
    Algorithm 6 in the submission: "Last-Iterate Convergence Properties of Regret-Matching Algorithms in Games."

    Input:
        T = number of iterations
        A = payoff matrix
        n,m = dimensions of simplices of each player
        q = exponent for the weights t**q
        R0 = initialization point
        
    Output:
        5 Arrays of dimension T:
            result_duality_gap: duality gap of current iterates
            results_x: x[t] at each iteration t=0,...T-1
            results_y: y[t] at each iteration t=0,...T-1
            results_regret_x: regret accumulated at each iteration for the x-player
            results_regret_y: regret accumulated at each iteration for the y-player
        
    '''
    
    A_T=np.transpose(A)
    
    result_duality_gap = np.zeros(T-1)
    results_regret_x = np.zeros(T-1)
    results_regret_y = np.zeros(T-1)
    results_x = np.zeros((T-1,n))
    results_y = np.zeros((T-1,m))
    
    # initialization
    x_tot=np.zeros(n)
    y_tot=np.zeros(m)    
    
    # running sum of iterates with uniform weights
    x_tot_unf=np.zeros(n)
    y_tot_unf=np.zeros(m)
    
    sum_instant_regret_x = 0
    sum_instant_regret_y = 0  
    
    z_R_x=np.maximum(np.zeros(n),R0*np.ones(n)) 
    z_R_y=np.maximum(np.zeros(m),R0*np.ones(m)) 
    
    z_R_x = (1/sum(z_R_x))*z_R_x
    z_R_y = (1/sum(z_R_y))*z_R_y
    
    w_R_x = z_R_x.copy()
    w_R_y = z_R_y.copy()
    
    x_loc = compute_next_decision(z_R_x)
    y_loc = compute_next_decision(z_R_y)
    
    c_x = eta*np.dot(A,y_loc)
    c_y = - eta*np.dot(A_T,x_loc)
    
    k_restarting = 0
    
    for t in range(1,T):
        
        # z_R : z^{t-1}
        # w_R : w^t

        z_R_x = projection_onto_chopped_orthant(compute_next_aggregate_payoff(w_R_x,x_loc,c_x))
        x_loc=compute_next_decision(z_R_x)
        
        # Compute next decision for y-player
        z_R_y = projection_onto_chopped_orthant(compute_next_aggregate_payoff(w_R_y,y_loc,c_y))
        y_loc=compute_next_decision(z_R_y)
        
        c_x = eta*np.dot(A,y_loc)
        c_y = - eta*np.dot(A_T,x_loc)
        
        # store a copy of these vectors
        # to check restarting conditions
        
        w_R_x_old = w_R_x.copy()
        w_R_y_old = w_R_y.copy()
        
        # Update aggregate payoffs
        w_R_x=projection_onto_chopped_orthant(compute_next_aggregate_payoff(w_R_x,x_loc,c_x))
        w_R_y=projection_onto_chopped_orthant(compute_next_aggregate_payoff(w_R_y,y_loc,c_y))
        
        x=compute_next_decision(w_R_x)
        y=compute_next_decision(w_R_y)
        
        sum_instant_regret_x += np.dot(x,c_x)
        sum_instant_regret_y += np.dot(y,c_y)
        
        # Update running sums of decisions
        x_tot += x
        y_tot += y
        
        x_tot_unf += x
        y_tot_unf += y
        
        # here:
            # z_R: z^t
            # w_R: w^{t+1}
            # w_R_old: w^t
        
        # checking if we need to restart:
        if np.sqrt(np.linalg.norm(w_R_x-z_R_x)**2 + np.linalg.norm(w_R_y-z_R_y)**2) + np.sqrt(np.linalg.norm(w_R_x_old-z_R_x)**2 + np.linalg.norm(w_R_y_old-z_R_y)**2) < 8/(2**k_restarting):
            k_restarting +=1
            z_R_x = x
            z_R_y = y
            w_R_x = x
            w_R_y = y
            x_loc = x
            y_loc = y
            c_x = eta*np.dot(A,y_loc)
            c_y = - eta*np.dot(A_T,x_loc)      
            
        # Compute duality gap for current iterates
        result_duality_gap[t-1]=np.max(np.dot(A_T,x))-np.min(np.dot(A,y))
        results_x[t-1]=x
        results_y[t-1]=y
        results_regret_x[t-1] = sum_instant_regret_x-np.min(np.dot(A,y_tot_unf))
        results_regret_y[t-1] = np.max(np.dot(A_T,x_tot_unf))+sum_instant_regret_y
        
    return result_duality_gap, results_x, results_y, results_regret_x, results_regret_y

##########################
#########
#########     Extragradient algorithm, optimistic OMD
#########
##########################

def extragradient_algorithm(T,A,n,m,eta): 
    
    '''
      
    Extragradient algorithm
    
    Algorithm 9 in the submission: "Last-Iterate Convergence Properties of Regret-Matching Algorithms in Games."
    
    Input:
        T = number of iterations
        A = payoff matrix
        n,m = dimensions of simplices of each player
        eta = stepsize

        
    Output:
        5 Arrays of dimension T:
            result_duality_gap: duality gap of current iterates
            results_x: x[t] at each iteration t=0,...T-1
            results_y: y[t] at each iteration t=0,...T-1
            results_regret_x: regret accumulated at each iteration for the x-player
            results_regret_y: regret accumulated at each iteration for the y-player
        
    
 
    
    '''
    


    A_T=np.transpose(A)
    
    result_duality_gap = np.zeros(T-1)
    results_regret_x = np.zeros(T-1)
    results_regret_y = np.zeros(T-1)
    results_x = np.zeros((T-1,n))
    results_y = np.zeros((T-1,m))
    
    # initialization
    x_tot=np.zeros(n)
    y_tot=np.zeros(m)    
    
    # running sum of iterates with uniform weights
    x_tot_unf=np.zeros(n)
    y_tot_unf=np.zeros(m)
    
    sum_instant_regret_x = 0
    sum_instant_regret_y = 0  
    
    x = (1/n)*np.ones(n)
    y = (1/m)*np.ones(m)
    
    c_x = eta*np.dot(A,y)
    c_y = - eta*np.dot(A_T,x)
    
    for t in range(1,T):

        # Notations from the paper (algorithm 9):
        # z[t+1/2] = x_h,y_h
        # z[t+1] = x,y

        sum_instant_regret_x += np.dot(x,c_x)
        sum_instant_regret_y += np.dot(y,c_y)
        
        c_x = - np.dot(A, y)
        x_h = euclidean_proj_simplex(x + eta*c_x)
            
        c_y = np.dot(A_T, x) 
        y_h = euclidean_proj_simplex(y + eta*c_y)

        c_x_h = - np.dot(A, y_h)
        x = euclidean_proj_simplex(x + eta*c_x_h)
            
        c_y_h = np.dot(A_T, x_h) 
        y = euclidean_proj_simplex(y + eta*c_y_h)
        
        # Update running sums of decisions
        x_tot += x
        y_tot += y
        
        x_tot_unf += x
        y_tot_unf += y
            
        # Compute duality gap for current iterates
        result_duality_gap[t-1]=np.max(np.dot(A_T,x))-np.min(np.dot(A,y))
        results_x[t-1]=x
        results_y[t-1]=y
        results_regret_x[t-1] = sum_instant_regret_x-np.min(np.dot(A,y_tot_unf))
        results_regret_y[t-1] = np.max(np.dot(A_T,x_tot_unf))+sum_instant_regret_y
        
    return result_duality_gap, results_x, results_y, results_regret_x, results_regret_y

def optimistic_omd(T,A,n,m,eta): #  run optimistic OMD on the simplex
    
    '''
      
    Optimistic Online Mirror Descent
    
    Algorithm 10 in the submission: "Last-Iterate Convergence Properties of Regret-Matching Algorithms in Games."
    
    Input:
        T = number of iterations
        A = payoff matrix
        n,m = dimensions of simplices of each player
        eta = stepsize

        
    Output:
        5 Arrays of dimension T:
            result_duality_gap: duality gap of current iterates
            results_x: x[t] at each iteration t=0,...T-1
            results_y: y[t] at each iteration t=0,...T-1
            results_regret_x: regret accumulated at each iteration for the x-player
            results_regret_y: regret accumulated at each iteration for the y-player
        
    
 
    
    '''

    A_T=np.transpose(A)
    
    result_duality_gap = np.zeros(T-1)
    results_regret_x = np.zeros(T-1)
    results_regret_y = np.zeros(T-1)
    results_x = np.zeros((T-1,n))
    results_y = np.zeros((T-1,m))
    
    # initialization
    x_tot=np.zeros(n)
    y_tot=np.zeros(m)    
    
    # running sum of iterates with uniform weights
    x_tot_unf=np.zeros(n)
    y_tot_unf=np.zeros(m)
    
    sum_instant_regret_x = 0
    sum_instant_regret_y = 0  
    
    x = (1/n)*np.ones(n)
    y = (1/m)*np.ones(m)

    x_h = (1/n)*np.ones(n)
    y_h = (1/m)*np.ones(m)
    
    c_x = - np.dot(A, y)
    c_y = np.dot(A_T, x) 

    
    for t in range(1,T):
        
        # Notations from the paper (algorithm 10):
        # w[t] = x_h,y_h
        # z[t] = x,y

        x = euclidean_proj_simplex(x_h + eta*c_x)
        y = euclidean_proj_simplex(y_h + eta*c_y)
        
        c_x = - np.dot(A, y)
        c_y = np.dot(A_T, x) 

        x_h = euclidean_proj_simplex(x_h + eta*c_x)
        y_h = euclidean_proj_simplex(y_h + eta*c_y)
        
        sum_instant_regret_x += np.dot(x,c_x)
        sum_instant_regret_y += np.dot(y,c_y)
        
        # Update running sums of decisions
        
        x_tot += x
        y_tot += y
        
        x_tot_unf += x
        y_tot_unf += y
            
        # Compute duality gap for current iterates
        result_duality_gap[t-1]=np.max(np.dot(A_T,x))-np.min(np.dot(A,y))
        results_x[t-1]=x
        results_y[t-1]=y
        results_regret_x[t-1] = sum_instant_regret_x-np.min(np.dot(A,y_tot_unf))
        results_regret_y[t-1] = np.max(np.dot(A_T,x_tot_unf))+sum_instant_regret_y
        
    return result_duality_gap, results_x, results_y, results_regret_x, results_regret_y


########################################################################################################
########################################################################################################
########################################################################################################

##########################
#########
#########   Simulation parameters and setup
#########
##########################


### choose an algorithm to run among the following:
    
# comment/outcomment the one to run
    
alg = 'rmp' # Regret Matching+
# alg = 'prmp' # Predictive Regret Matching+
# alg = 'alt_rmp' # Alternating Regret Matching+
# alg = 'alt_prmp' # Alternating Predictive Regret Matching+
# alg = 'crmp' # ExRM+
# alg = 'crmp_rs' # RS-ExRM+
# alg = 'smooth_prmp' # Smooth Predictive Regret Matching+ (SPRM+)
# alg = 'smooth_prmp_rs' # RS-SPRM+
# alg = 'extragradient_alg' # Extragradient algorithm
# alg = 'optimistic_omd' # Optimistic OMD

### Choose a matrix game instance: small matrix, random matrix games, Kuhn pojer, Goofspiel poker

# run simulation on our pathological small 3x3 matrix game instance
# Return/plot/save: the duality gap, the regrets for both players, and the difference in the last consecutive iterates
run_sim_small_matrix = 1

# run simulation on 10x15 matrix games generated at random with normal distributions for the coefficients
# this runs the following algorithms: RM+, PRM+, ExRM+, Smooth PRM+, RS-ExRM+, RS-SPRM+
run_random_sim = 0
# In case we want to run simulations with random matrices and with only one algorithm chosen above:
run_random_sim_one_alg = 0

# run simulation on Kuhn poker
run_kuhn = 0

# run simulations on Goofspiel 
run_goofspiel = 0

### Parameters for the simulations

# Number of iterations
T=10000
# Number of fixed point iterations for Conceptual RM+; k=2 -> ExRM+
k=2
# Number of random seeds for experiments with random matrix games
nb_exp=25
# Exponents for the weights on decisions in the average of the iterates; q=1 -> linear averaging, q=0 -> uniform averaging
q=1
# Step size
eta = 0.01
# Initialization point
R0 = 1
# Number of points to consider for plotting the last || x_t - x_t+1 || and || y_t - y_t+1 ||
nb_last_iterate = 100

# Number of points on figure
nb_points=30

##########################
#########
#########   Run the algorithms
#########
##########################

if run_sim_small_matrix:
    
    # 3x3 matrix instance
    A = [[3,0,-3],[0,3,-4],[0,0,1]]
    
    n=len(A[0])
    m=n
    start = time.time()
    if alg == 'rmp':
        print('running RM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y=regret_matching_plus(T,A,n,m,q,R0)    
    elif alg == 'alt_rmp':
        print('running alt-RM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y=regret_matching_alt_plus(T,A,n,m,q,R0) 
    elif alg == 'prmp':
        print('running PRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y=optimistic_regret_matching_plus(T,A,n,m,q,R0)   
    elif alg == 'alt_prmp':
        print('running alt-PRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_regret_matching_plus_alt(T,A,n,m,q,R0)  
    elif alg == 'crmp':
        print('running ExRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = conceptual_regret_matching_plus_chopped(T,A,n,m,q,k,R0,eta)   
    elif alg == 'smooth_prmp':
        print('running smooth-PRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_regret_matching_plus_chopped(T,A,n,m,q,R0,eta)
    elif alg == 'alt_smooth_prmp':
        print('running alt-smooth-PRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_regret_matching_plus_alt_chopped(T,A,n,m,q,R0,eta)
    elif alg == 'crmp_rs':
        print('running ExRM+ with restarting...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = conceptual_regret_matching_plus_chopped_restarting(T,A,n,m,q,k,R0,eta)   
    elif alg == 'smooth_prmp_rs':
        print('running Smooth PRM+ with restarting...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_regret_matching_plus_chopped_restarting(T,A,n,m,q,R0,eta)
    elif alg == 'extragradient_alg':
        print('running Extragradient Algorithm...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = extragradient_algorithm(T,A,n,m,eta)
    elif alg == 'optimistic_omd':
        print('running Optimistic OMD...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_omd(T,A,n,m,eta)              
        
    end = time.time()
    
    print('Running time:', np.around(end-start,2))
    
    np.save(f'numerical_results/{alg}_duality_gap_{T}_eta_{eta}.npy',result_duality_gap)        
    np.save(f'numerical_results/{alg}_results_x_{T}_eta_{eta}.npy',results_x)        
    np.save(f'numerical_results/{alg}_results_y_{T}_eta_{eta}.npy',results_y)        
    np.save(f'numerical_results/{alg}_regret_x_{T}_eta_{eta}.npy',regret_x)        
    np.save(f'numerical_results/{alg}_regret_y_{T}_eta_{eta}.npy',regret_y) 
    
    x_axis = np.asarray(range(T-1))
    indices = np.logspace(0.0, np.log10(len(x_axis)), num=nb_points)
    indices=indices.astype(int)
    indices[-1]=indices[-1]-2
    
    ######### Plot the duality gap

    fig, ax = plt.subplots()
    
    l1 =  ax.plot(x_axis[indices], result_duality_gap[indices], color='red', marker='*', linestyle=':',linewidth=2, markersize=10,label = 'Duality gap')

    plt.xscale('log')
    plt.yscale('log')
    
    ax.set_xlabel('Number of iterations',fontsize=14)
    ax.set_ylabel('Duality gap',fontsize=14)
    ax.tick_params(axis='both', which='major', labelsize=14)

    ax.grid()
    
    fig.legend(ncol=2,loc="upper center",bbox_to_anchor=(0.5,1.2,0,0),fontsize=15)
            
    fig.savefig(f"figures/duality_gap_T_{T}_{alg}.pdf", bbox_inches='tight')
    plt.show()
      
if run_random_sim:
    
    # players dimensions
    n=10
    m=15
    
    # Initialization point for ExRM+/Smooth PRM+: need to ensure that the initial point belong to the stable region
    
    distribution_list = ['normal']
            
    for distribution in distribution_list:
        print('Running simulations with random instances ...')
        
        y_RM_plus=np.zeros((nb_exp,T-1)) 
        y_PRM_plus=np.zeros((nb_exp,T-1)) 
        y_CRM_plus=np.zeros((nb_exp,T-1))
        y_PRM_plus_chop=np.zeros((nb_exp,T-1))
        y_CRM_plus_rs=np.zeros((nb_exp,T-1))
        y_PRM_plus_chop_rs=np.zeros((nb_exp,T-1))
        
        for i in range(nb_exp):
            
            print('iteration: ',i)
                
            np.random.seed(i)
            
            if distribution == 'uniform':
            
                A=np.random.rand(n,m)
            
            elif distribution == 'normal':
            
                A=np.random.normal(loc=0,scale=1,size=(n,m))
            
            result_duality_gap, results_x, results_y, regret_x, regret_y=regret_matching_plus(T,A,n,m,q,R0)
            y_axis_1 = result_duality_gap 
            result_duality_gap, results_x, results_y, regret_x, regret_y=optimistic_regret_matching_plus(T,A,n,m,q,R0)
            y_axis_2 = result_duality_gap 
            result_duality_gap, results_x, results_y, regret_x, regret_y=conceptual_regret_matching_plus_chopped(T,A,n,m,q,k,R0,eta)
            y_axis_3 = result_duality_gap 
            result_duality_gap, results_x, results_y, regret_x, regret_y=optimistic_regret_matching_plus_chopped(T,A,n,m,q,R0,eta)
            y_axis_4 = result_duality_gap 
            result_duality_gap, results_x, results_y, regret_x, regret_y=conceptual_regret_matching_plus_chopped_restarting(T,A,n,m,q,k,R0,eta)
            y_axis_5 = result_duality_gap 
            result_duality_gap, results_x, results_y, regret_x, regret_y=optimistic_regret_matching_plus_chopped_restarting(T,A,n,m,q,R0,eta)
            y_axis_6 = result_duality_gap 
            
            y_RM_plus[i,:]=y_axis_1
            y_PRM_plus[i,:]=y_axis_2
            y_CRM_plus[i,:]=y_axis_3
            y_PRM_plus_chop[i,:]=y_axis_4
            y_CRM_plus_rs[i,:]=y_axis_5
            y_PRM_plus_chop_rs[i,:]=y_axis_6
            
        y_RM_plus_mean=np.mean(y_RM_plus,axis=0)
        y_PRM_plus_mean=np.mean(y_PRM_plus,axis=0)
        y_CRM_plus_mean=np.mean(y_CRM_plus,axis=0)
        y_PRM_plus_chop_mean=np.mean(y_PRM_plus_chop,axis=0)
        y_CRM_plus_rs_mean=np.mean(y_CRM_plus_rs,axis=0)
        y_PRM_plus_chop_rs_mean=np.mean(y_PRM_plus_chop_rs,axis=0)
        
        y_RM_plus_ci = (1.96/np.sqrt(nb_exp))*np.std(y_RM_plus,axis=0)
        y_PRM_plus_ci = (1.96/np.sqrt(nb_exp))*np.std(y_PRM_plus,axis=0)
        y_CRM_plus_ci = (1.96/np.sqrt(nb_exp))*np.std(y_CRM_plus,axis=0)
        y_PRM_plus_chop_ci = (1.96/np.sqrt(nb_exp))*np.std(y_PRM_plus_chop,axis=0)
        y_CRM_plus_rs_ci = (1.96/np.sqrt(nb_exp))*np.std(y_CRM_plus_rs,axis=0)
        y_PRM_plus_chop_rs_ci = (1.96/np.sqrt(nb_exp))*np.std(y_PRM_plus_chop_rs,axis=0)
        
        indices = np.logspace(0.0, np.log10(T), num=nb_points)
        indices=indices.astype(int)
        indices[-1]=indices[-1]-2 
        
        x_axis=np.asarray(range(T-1))
        x_axis=x_axis[indices]
        
        y_RM_plus_mean = y_RM_plus_mean[indices]
        y_PRM_plus_mean = y_PRM_plus_mean[indices]
        y_CRM_plus_mean = y_CRM_plus_mean[indices]
        y_PRM_plus_chop_mean = y_PRM_plus_chop_mean[indices]
        y_CRM_plus_rs_mean = y_CRM_plus_rs_mean[indices]
        y_PRM_plus_chop_rs_mean = y_PRM_plus_chop_rs_mean[indices]

        y_RM_plus_ci = y_RM_plus_ci[indices]
        y_PRM_plus_ci = y_PRM_plus_ci[indices]
        y_CRM_plus_ci = y_CRM_plus_ci[indices]
        y_PRM_plus_chop_ci = y_PRM_plus_chop_ci[indices]  
        y_CRM_plus_rs_ci = y_CRM_plus_rs_ci[indices]
        y_PRM_plus_chop_rs_ci = y_PRM_plus_chop_rs_ci[indices]  
        
        fig, ax = plt.subplots()
        
        l1=  ax.plot(x_axis, y_RM_plus_mean, color='green', marker='o', linestyle='-', linewidth=2, markersize=7,label=f'RM$^+$')
        l2=  ax.plot(x_axis, y_PRM_plus_mean, color='red', marker='*', linestyle=':',linewidth=2, markersize=7,label=f'PRM$^+$')
        l3=  ax.plot(x_axis, y_CRM_plus_mean, color='black', marker='s', linestyle='--',linewidth=2, markersize=7,label=f'CRM$^+$')
        l4=  ax.plot(x_axis, y_PRM_plus_chop_mean, color='blue', marker='D', linestyle='--',linewidth=2, markersize=7,label=f'Smooth PRM$^+$ ')
        
        plt.fill_between(x_axis, y_RM_plus_mean-y_RM_plus_ci,y_RM_plus_mean+y_RM_plus_ci, color='green', alpha=0.1)
        plt.fill_between(x_axis, y_PRM_plus_mean-y_PRM_plus_ci,y_PRM_plus_mean+y_PRM_plus_ci, color='red', alpha=0.1)
        plt.fill_between(x_axis, y_CRM_plus_mean-y_CRM_plus_ci,y_CRM_plus_mean+y_CRM_plus_ci, color='black', alpha=0.1)
        plt.fill_between(x_axis, y_PRM_plus_chop_mean-y_PRM_plus_chop_ci,y_PRM_plus_chop_mean+y_PRM_plus_chop_ci, color='blue', alpha=0.1)

        # save the duality gap averaged across random games and the confidence intervals

        alg = 'rmp'
        
        np.save(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}.npy',y_RM_plus_mean)
        np.save(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.npy',y_RM_plus_ci)

        alg = 'prmp'
        
        np.save(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}.npy',y_PRM_plus_mean)
        np.save(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.npy',y_PRM_plus_ci)


        alg = 'crmp'
        
        np.save(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}.npy',y_CRM_plus_mean)
        np.save(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.npy',y_CRM_plus_ci)

        alg = 'smooth_prmp'
        
        np.save(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}.npy',y_PRM_plus_chop_mean)
        np.save(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.npy',y_PRM_plus_chop_ci)

        alg = 'crmp_rs'
        
        np.save(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}.npy',y_CRM_plus_rs_mean)
        np.save(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.npy',y_CRM_plus_rs_ci)

        alg = 'smooth_prmp_rs'
        
        np.save(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}.npy',y_PRM_plus_chop_rs_mean)
        np.save(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.npy',y_PRM_plus_chop_rs_ci)

        plt.xscale('log')
        plt.yscale('log')
        
        ax.set_xlabel('Number of iterations',fontsize=20)
        ax.set_ylabel('Duality gap',fontsize=20)
        ax.tick_params(axis='both', which='major', labelsize=20)

        ax.grid()
        
        fig.legend(ncol=5,loc="upper center")
                
        plt.show()

if run_random_sim_one_alg:
    
    # players dimensions
    n=10
    m=15
    
    distribution_list = ['normal']
            
    for distribution in distribution_list:
        print('Running simulations with random instances ...')
        
        y_axis=np.zeros((nb_exp,T-1)) 
        
        for i in range(nb_exp):
            
            print('iteration: ',i)
                
            np.random.seed(i)
            
            if distribution == 'uniform':
            
                A=np.random.rand(n,m)
            
            elif distribution == 'normal':
            
                A=np.random.normal(loc=0,scale=1,size=(n,m))
            
            if alg == 'rmp':
                print('running RM+...')
                result_duality_gap, results_x, results_y, regret_x, regret_y=regret_matching_plus(T,A,n,m,q,R0)    
            elif alg == 'alt_rmp':
                print('running alt-RM+...')
                result_duality_gap, results_x, results_y, regret_x, regret_y=regret_matching_alt_plus(T,A,n,m,q,R0) 
            elif alg == 'prmp':
                print('running PRM+...')
                result_duality_gap, results_x, results_y, regret_x, regret_y=optimistic_regret_matching_plus(T,A,n,m,q,R0)   
            elif alg == 'alt_prmp':
                print('running alt-PRM+...')
                result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_regret_matching_plus_alt(T,A,n,m,q,R0)  
            elif alg == 'crmp':
                print('running ExRM+...')
                result_duality_gap, results_x, results_y, regret_x, regret_y = conceptual_regret_matching_plus_chopped(T,A,n,m,q,k,R0,eta)   
            elif alg == 'smooth_prmp':
                print('running smooth-PRM+...')
                result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_regret_matching_plus_chopped(T,A,n,m,q,R0,eta)
            elif alg == 'alt_smooth_prmp':
                print('running alt-smooth-PRM+...')
                result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_regret_matching_plus_alt_chopped(T,A,n,m,q,R0,eta)
            elif alg == 'crmp_rs':
                print('running ExRM+ with restarting...')
                result_duality_gap, results_x, results_y, regret_x, regret_y = conceptual_regret_matching_plus_chopped_restarting(T,A,n,m,q,k,R0,eta)   
            elif alg == 'smooth_prmp_rs':
                print('running Smooth PRM+ with restarting...')
                result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_regret_matching_plus_chopped_restarting(T,A,n,m,q,R0,eta)
            elif alg == 'extragradient_alg':
                print('running extragradient_algorithm...')
                result_duality_gap, results_x, results_y, regret_x, regret_y = extragradient_algorithm(T,A,n,m,eta)
            elif alg == 'optimistic_omd':
                print('running optimistic_omd...')
                result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_omd(T,A,n,m,eta)
                         
            y_axis[i,:]=result_duality_gap
            
        y_axis_mean=np.mean(y_axis,axis=0)
        y_axis_ci = (1.96/np.sqrt(nb_exp))*np.std(y_axis,axis=0)
        
        indices = np.logspace(0.0, np.log10(T), num=nb_points)
        indices=indices.astype(int)
        indices[-1]=indices[-1]-2 
        
        x_axis=np.asarray(range(T-1))
        x_axis=x_axis[indices]
        
        y_axis_mean = y_axis_mean[indices]
        y_axis_ci = y_axis_ci[indices]
        
        
        fig, ax = plt.subplots()
        
        l1=  ax.plot(x_axis, y_axis_mean, color='green', marker='o', linestyle='-', linewidth=2, markersize=7,label=f'{alg}')

        
        plt.fill_between(x_axis, y_axis_mean-y_axis_ci,y_axis_mean+y_axis_ci, color='green', alpha=0.1)

        np.save(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}.npy',y_axis_mean)
        np.save(f'numerical_results/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.npy',y_axis_ci)

        plt.xscale('log')
        plt.yscale('log')
        
        ax.set_xlabel('Number of iterations',fontsize=20)
        ax.set_ylabel('Duality gap',fontsize=20)
        ax.tick_params(axis='both', which='major', labelsize=20)

        ax.grid()
        
        fig.legend(ncol=5,loc="upper center")
        plt.savefig(f'figures/{alg}_duality_gap_{T}_random_size_{n}_{m}_eta_{eta}_ci.pdf', bbox_inches='tight')       
        plt.show()
    
if run_kuhn:
    
    # Define matrix of the game
    A = [
    [+0.00e+00, +0.00e+00, +3.33e-01, +3.33e-01, +0.00e+00, +0.00e+00, +3.33e-01, +3.33e-01, +0.00e+00, +0.00e+00, +3.33e-01, +3.33e-01, +0.00e+00, +0.00e+00, +3.33e-01, +3.33e-01, +0.00e+00, +0.00e+00, +3.33e-01, +3.33e-01, +0.00e+00, +0.00e+00, +3.33e-01, +3.33e-01, +0.00e+00, +0.00e+00, +3.33e-01, +3.33e-01, +0.00e+00, +0.00e+00, +3.33e-01, +3.33e-01, -3.33e-01, -3.33e-01, +0.00e+00, +0.00e+00, -3.33e-01, -3.33e-01, +0.00e+00, +0.00e+00, -3.33e-01, -3.33e-01, +0.00e+00, +0.00e+00, -3.33e-01, -3.33e-01, +0.00e+00, +0.00e+00, -3.33e-01, -3.33e-01, +0.00e+00, +0.00e+00, -3.33e-01, -3.33e-01, +0.00e+00, +0.00e+00, -3.33e-01, -3.33e-01, +0.00e+00, +0.00e+00, -3.33e-01, -3.33e-01, +0.00e+00, +0.00e+00, ],
    [+0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, ],
    [+0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, +5.00e-01, +3.33e-01, +6.67e-01, +5.00e-01, +5.00e-01, +3.33e-01, +6.67e-01, +5.00e-01, +5.00e-01, +3.33e-01, +6.67e-01, +5.00e-01, +5.00e-01, +3.33e-01, +6.67e-01, +5.00e-01, -1.67e-01, -3.33e-01, +0.00e+00, -1.67e-01, -1.67e-01, -3.33e-01, +0.00e+00, -1.67e-01, -1.67e-01, -3.33e-01, +0.00e+00, -1.67e-01, -1.67e-01, -3.33e-01, +0.00e+00, -1.67e-01, +3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, +3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, +3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, +3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, ],
    [+0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, -5.00e-01, -5.00e-01, -6.67e-01, -6.67e-01, -5.00e-01, -5.00e-01, -6.67e-01, -6.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, -5.00e-01, -5.00e-01, -6.67e-01, -6.67e-01, -5.00e-01, -5.00e-01, -6.67e-01, -6.67e-01, -3.33e-01, -3.33e-01, -5.00e-01, -5.00e-01, -3.33e-01, -3.33e-01, -5.00e-01, -5.00e-01, -8.33e-01, -8.33e-01, -1.00e+00, -1.00e+00, -8.33e-01, -8.33e-01, -1.00e+00, -1.00e+00, -3.33e-01, -3.33e-01, -5.00e-01, -5.00e-01, -3.33e-01, -3.33e-01, -5.00e-01, -5.00e-01, -8.33e-01, -8.33e-01, -1.00e+00, -1.00e+00, -8.33e-01, -8.33e-01, -1.00e+00, -1.00e+00, ],
    [+0.00e+00, +0.00e+00, -6.67e-01, -6.67e-01, +0.00e+00, +0.00e+00, -6.67e-01, -6.67e-01, -5.00e-01, -5.00e-01, -1.17e+00, -1.17e+00, -5.00e-01, -5.00e-01, -1.17e+00, -1.17e+00, +0.00e+00, +0.00e+00, -6.67e-01, -6.67e-01, +0.00e+00, +0.00e+00, -6.67e-01, -6.67e-01, -5.00e-01, -5.00e-01, -1.17e+00, -1.17e+00, -5.00e-01, -5.00e-01, -1.17e+00, -1.17e+00, -1.67e-01, -1.67e-01, -8.33e-01, -8.33e-01, -1.67e-01, -1.67e-01, -8.33e-01, -8.33e-01, -6.67e-01, -6.67e-01, -1.33e+00, -1.33e+00, -6.67e-01, -6.67e-01, -1.33e+00, -1.33e+00, -1.67e-01, -1.67e-01, -8.33e-01, -8.33e-01, -1.67e-01, -1.67e-01, -8.33e-01, -8.33e-01, -6.67e-01, -6.67e-01, -1.33e+00, -1.33e+00, -6.67e-01, -6.67e-01, -1.33e+00, -1.33e+00, ],
    [+0.00e+00, -1.67e-01, -3.33e-01, -5.00e-01, +0.00e+00, -1.67e-01, -3.33e-01, -5.00e-01, -5.00e-01, -6.67e-01, -8.33e-01, -1.00e+00, -5.00e-01, -6.67e-01, -8.33e-01, -1.00e+00, +5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, +5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, -5.55e-17, -1.67e-01, -3.33e-01, -5.00e-01, -5.55e-17, -1.67e-01, -3.33e-01, -5.00e-01, -1.67e-01, -3.33e-01, -5.00e-01, -6.67e-01, -1.67e-01, -3.33e-01, -5.00e-01, -6.67e-01, -6.67e-01, -8.33e-01, -1.00e+00, -1.17e+00, -6.67e-01, -8.33e-01, -1.00e+00, -1.17e+00, +3.33e-01, +1.67e-01, -5.55e-17, -1.67e-01, +3.33e-01, +1.67e-01, -5.55e-17, -1.67e-01, -1.67e-01, -3.33e-01, -5.00e-01, -6.67e-01, -1.67e-01, -3.33e-01, -5.00e-01, -6.67e-01, ],
    [+3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, +0.00e+00, -1.67e-01, -1.67e-01, -3.33e-01, +0.00e+00, -1.67e-01, -3.33e-01, -5.00e-01, -1.67e-01, -3.33e-01, +0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, +0.00e+00, -1.67e-01, -1.67e-01, -3.33e-01, +0.00e+00, -1.67e-01, -3.33e-01, -5.00e-01, -1.67e-01, -3.33e-01, ],
    [+3.33e-01, +1.67e-01, -2.78e-17, -1.67e-01, +1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, +0.00e+00, -1.67e-01, -3.33e-01, -5.00e-01, +3.33e-01, +1.67e-01, -2.78e-17, -1.67e-01, +1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, +0.00e+00, -1.67e-01, -3.33e-01, -5.00e-01, +1.67e-01, -5.55e-17, -1.67e-01, -3.33e-01, -5.55e-17, -1.67e-01, -3.33e-01, -5.00e-01, +0.00e+00, -1.67e-01, -3.33e-01, -5.00e-01, -1.67e-01, -3.33e-01, -5.00e-01, -6.67e-01, +1.67e-01, -5.55e-17, -1.67e-01, -3.33e-01, -5.55e-17, -1.67e-01, -3.33e-01, -5.00e-01, +0.00e+00, -1.67e-01, -3.33e-01, -5.00e-01, -1.67e-01, -3.33e-01, -5.00e-01, -6.67e-01, ],
    [+3.33e-01, +0.00e+00, +3.33e-01, +0.00e+00, +1.67e-01, -1.67e-01, +1.67e-01, -1.67e-01, +1.67e-01, -1.67e-01, +1.67e-01, -1.67e-01, +0.00e+00, -3.33e-01, +0.00e+00, -3.33e-01, +8.33e-01, +5.00e-01, +8.33e-01, +5.00e-01, +6.67e-01, +3.33e-01, +6.67e-01, +3.33e-01, +6.67e-01, +3.33e-01, +6.67e-01, +3.33e-01, +5.00e-01, +1.67e-01, +5.00e-01, +1.67e-01, +1.67e-01, -1.67e-01, +1.67e-01, -1.67e-01, +0.00e+00, -3.33e-01, +0.00e+00, -3.33e-01, +1.11e-16, -3.33e-01, +1.11e-16, -3.33e-01, -1.67e-01, -5.00e-01, -1.67e-01, -5.00e-01, +6.67e-01, +3.33e-01, +6.67e-01, +3.33e-01, +5.00e-01, +1.67e-01, +5.00e-01, +1.67e-01, +5.00e-01, +1.67e-01, +5.00e-01, +1.67e-01, +3.33e-01, +0.00e+00, +3.33e-01, +0.00e+00, ],
    [+0.00e+00, +0.00e+00, +3.33e-01, +3.33e-01, +0.00e+00, +0.00e+00, +3.33e-01, +3.33e-01, +1.67e-01, +1.67e-01, +5.00e-01, +5.00e-01, +1.67e-01, +1.67e-01, +5.00e-01, +5.00e-01, +0.00e+00, +0.00e+00, +3.33e-01, +3.33e-01, +0.00e+00, +0.00e+00, +3.33e-01, +3.33e-01, +1.67e-01, +1.67e-01, +5.00e-01, +5.00e-01, +1.67e-01, +1.67e-01, +5.00e-01, +5.00e-01, -1.67e-01, -1.67e-01, +1.67e-01, +1.67e-01, -1.67e-01, -1.67e-01, +1.67e-01, +1.67e-01, -5.55e-17, -5.55e-17, +3.33e-01, +3.33e-01, -5.55e-17, -5.55e-17, +3.33e-01, +3.33e-01, -1.67e-01, -1.67e-01, +1.67e-01, +1.67e-01, -1.67e-01, -1.67e-01, +1.67e-01, +1.67e-01, -5.55e-17, -5.55e-17, +3.33e-01, +3.33e-01, -5.55e-17, -5.55e-17, +3.33e-01, +3.33e-01, ],
    [+0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, +1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, +1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, +1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, +1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, ],
    [+0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, +6.67e-01, +5.00e-01, +5.00e-01, +3.33e-01, +6.67e-01, +5.00e-01, +6.67e-01, +5.00e-01, +8.33e-01, +6.67e-01, +6.67e-01, +5.00e-01, +8.33e-01, +6.67e-01, +0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, +6.67e-01, +5.00e-01, +5.00e-01, +3.33e-01, +6.67e-01, +5.00e-01, +6.67e-01, +5.00e-01, +8.33e-01, +6.67e-01, +6.67e-01, +5.00e-01, +8.33e-01, +6.67e-01, ],
    [+0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, -5.00e-01, -5.00e-01, -3.33e-01, -3.33e-01, -5.00e-01, -5.00e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, -5.00e-01, -5.00e-01, -3.33e-01, -3.33e-01, -5.00e-01, -5.00e-01, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, -5.00e-01, -5.00e-01, -6.67e-01, -6.67e-01, -5.00e-01, -5.00e-01, -6.67e-01, -6.67e-01, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, -5.00e-01, -5.00e-01, -6.67e-01, -6.67e-01, -5.00e-01, -5.00e-01, -6.67e-01, -6.67e-01, ],
    [+0.00e+00, +0.00e+00, -6.67e-01, -6.67e-01, +0.00e+00, +0.00e+00, -6.67e-01, -6.67e-01, -3.33e-01, -3.33e-01, -1.00e+00, -1.00e+00, -3.33e-01, -3.33e-01, -1.00e+00, -1.00e+00, +0.00e+00, +0.00e+00, -6.67e-01, -6.67e-01, +0.00e+00, +0.00e+00, -6.67e-01, -6.67e-01, -3.33e-01, -3.33e-01, -1.00e+00, -1.00e+00, -3.33e-01, -3.33e-01, -1.00e+00, -1.00e+00, +0.00e+00, +0.00e+00, -6.67e-01, -6.67e-01, +0.00e+00, +0.00e+00, -6.67e-01, -6.67e-01, -3.33e-01, -3.33e-01, -1.00e+00, -1.00e+00, -3.33e-01, -3.33e-01, -1.00e+00, -1.00e+00, +0.00e+00, +0.00e+00, -6.67e-01, -6.67e-01, +0.00e+00, +0.00e+00, -6.67e-01, -6.67e-01, -3.33e-01, -3.33e-01, -1.00e+00, -1.00e+00, -3.33e-01, -3.33e-01, -1.00e+00, -1.00e+00, ],
    [+0.00e+00, -1.67e-01, -3.33e-01, -5.00e-01, +0.00e+00, -1.67e-01, -3.33e-01, -5.00e-01, -3.33e-01, -5.00e-01, -6.67e-01, -8.33e-01, -3.33e-01, -5.00e-01, -6.67e-01, -8.33e-01, +5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, +5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, +1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, +0.00e+00, -1.67e-01, -3.33e-01, -5.00e-01, +0.00e+00, -1.67e-01, -3.33e-01, -5.00e-01, -3.33e-01, -5.00e-01, -6.67e-01, -8.33e-01, -3.33e-01, -5.00e-01, -6.67e-01, -8.33e-01, +5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, +5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, +1.67e-01, -2.78e-17, -1.67e-01, -3.33e-01, +1.67e-01, -2.78e-17, -1.67e-01, -3.33e-01, ],
    [+3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, +3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, +3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, +3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, +1.67e-01, -5.55e-17, +3.33e-01, +1.67e-01, -5.55e-17, -1.67e-01, +1.67e-01, -2.78e-17, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, +1.67e-01, -5.55e-17, +1.67e-01, -5.55e-17, +3.33e-01, +1.67e-01, -5.55e-17, -1.67e-01, +1.67e-01, -2.78e-17, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, +1.67e-01, -5.55e-17, ],
    [+3.33e-01, +1.67e-01, -2.78e-17, -1.67e-01, +1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, +3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, +3.33e-01, +1.67e-01, -2.78e-17, -1.67e-01, +1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, +3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, +3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, +3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, +1.67e-01, -2.78e-17, -1.67e-01, -3.33e-01, +3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, +3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, +1.67e-01, -2.78e-17, -1.67e-01, -3.33e-01, ],
    [+3.33e-01, +0.00e+00, +3.33e-01, +0.00e+00, +1.67e-01, -1.67e-01, +1.67e-01, -1.67e-01, +3.33e-01, +0.00e+00, +3.33e-01, +0.00e+00, +1.67e-01, -1.67e-01, +1.67e-01, -1.67e-01, +8.33e-01, +5.00e-01, +8.33e-01, +5.00e-01, +6.67e-01, +3.33e-01, +6.67e-01, +3.33e-01, +8.33e-01, +5.00e-01, +8.33e-01, +5.00e-01, +6.67e-01, +3.33e-01, +6.67e-01, +3.33e-01, +3.33e-01, +0.00e+00, +3.33e-01, +0.00e+00, +1.67e-01, -1.67e-01, +1.67e-01, -1.67e-01, +3.33e-01, +0.00e+00, +3.33e-01, +0.00e+00, +1.67e-01, -1.67e-01, +1.67e-01, -1.67e-01, +8.33e-01, +5.00e-01, +8.33e-01, +5.00e-01, +6.67e-01, +3.33e-01, +6.67e-01, +3.33e-01, +8.33e-01, +5.00e-01, +8.33e-01, +5.00e-01, +6.67e-01, +3.33e-01, +6.67e-01, +3.33e-01, ],
    [-3.33e-01, -3.33e-01, +0.00e+00, +0.00e+00, +1.67e-01, +1.67e-01, +5.00e-01, +5.00e-01, -1.67e-01, -1.67e-01, +1.67e-01, +1.67e-01, +3.33e-01, +3.33e-01, +6.67e-01, +6.67e-01, +1.67e-01, +1.67e-01, +5.00e-01, +5.00e-01, +6.67e-01, +6.67e-01, +1.00e+00, +1.00e+00, +3.33e-01, +3.33e-01, +6.67e-01, +6.67e-01, +8.33e-01, +8.33e-01, +1.17e+00, +1.17e+00, -5.00e-01, -5.00e-01, -1.67e-01, -1.67e-01, +2.78e-17, +2.78e-17, +3.33e-01, +3.33e-01, -3.33e-01, -3.33e-01, +1.11e-16, +1.11e-16, +1.67e-01, +1.67e-01, +5.00e-01, +5.00e-01, +2.78e-17, +2.78e-17, +3.33e-01, +3.33e-01, +5.00e-01, +5.00e-01, +8.33e-01, +8.33e-01, +1.67e-01, +1.67e-01, +5.00e-01, +5.00e-01, +6.67e-01, +6.67e-01, +1.00e+00, +1.00e+00, ],
    [-3.33e-01, -3.33e-01, -5.00e-01, -5.00e-01, +1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, +3.33e-01, +3.33e-01, +1.67e-01, +1.67e-01, +1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, +6.67e-01, +6.67e-01, +5.00e-01, +5.00e-01, +3.33e-01, +3.33e-01, +1.67e-01, +1.67e-01, +8.33e-01, +8.33e-01, +6.67e-01, +6.67e-01, -3.33e-01, -3.33e-01, -5.00e-01, -5.00e-01, +1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, +3.33e-01, +3.33e-01, +1.67e-01, +1.67e-01, +1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, +6.67e-01, +6.67e-01, +5.00e-01, +5.00e-01, +3.33e-01, +3.33e-01, +1.67e-01, +1.67e-01, +8.33e-01, +8.33e-01, +6.67e-01, +6.67e-01, ],
    [-3.33e-01, -5.00e-01, -1.67e-01, -3.33e-01, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, -1.67e-01, -3.33e-01, +0.00e+00, -1.67e-01, +3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, +6.67e-01, +5.00e-01, +8.33e-01, +6.67e-01, +1.17e+00, +1.00e+00, +1.33e+00, +1.17e+00, +8.33e-01, +6.67e-01, +1.00e+00, +8.33e-01, +1.33e+00, +1.17e+00, +1.50e+00, +1.33e+00, -3.33e-01, -5.00e-01, -1.67e-01, -3.33e-01, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, -1.67e-01, -3.33e-01, +0.00e+00, -1.67e-01, +3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, +6.67e-01, +5.00e-01, +8.33e-01, +6.67e-01, +1.17e+00, +1.00e+00, +1.33e+00, +1.17e+00, +8.33e-01, +6.67e-01, +1.00e+00, +8.33e-01, +1.33e+00, +1.17e+00, +1.50e+00, +1.33e+00, ],
    [-3.33e-01, -3.33e-01, -5.00e-01, -5.00e-01, +1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, -6.67e-01, -6.67e-01, -8.33e-01, -8.33e-01, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, +1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, +6.67e-01, +6.67e-01, +5.00e-01, +5.00e-01, -1.67e-01, -1.67e-01, -3.33e-01, -3.33e-01, +3.33e-01, +3.33e-01, +1.67e-01, +1.67e-01, -5.00e-01, -5.00e-01, -6.67e-01, -6.67e-01, +2.78e-17, +2.78e-17, -1.67e-01, -1.67e-01, -8.33e-01, -8.33e-01, -1.00e+00, -1.00e+00, -3.33e-01, -3.33e-01, -5.00e-01, -5.00e-01, +2.78e-17, +2.78e-17, -1.67e-01, -1.67e-01, +5.00e-01, +5.00e-01, +3.33e-01, +3.33e-01, -3.33e-01, -3.33e-01, -5.00e-01, -5.00e-01, +1.67e-01, +1.67e-01, +0.00e+00, +0.00e+00, ],
    [-3.33e-01, -3.33e-01, -1.00e+00, -1.00e+00, +1.67e-01, +1.67e-01, -5.00e-01, -5.00e-01, -6.67e-01, -6.67e-01, -1.33e+00, -1.33e+00, -1.67e-01, -1.67e-01, -8.33e-01, -8.33e-01, +1.67e-01, +1.67e-01, -5.00e-01, -5.00e-01, +6.67e-01, +6.67e-01, +0.00e+00, +0.00e+00, -1.67e-01, -1.67e-01, -8.33e-01, -8.33e-01, +3.33e-01, +3.33e-01, -3.33e-01, -3.33e-01, -3.33e-01, -3.33e-01, -1.00e+00, -1.00e+00, +1.67e-01, +1.67e-01, -5.00e-01, -5.00e-01, -6.67e-01, -6.67e-01, -1.33e+00, -1.33e+00, -1.67e-01, -1.67e-01, -8.33e-01, -8.33e-01, +1.67e-01, +1.67e-01, -5.00e-01, -5.00e-01, +6.67e-01, +6.67e-01, -2.78e-17, -2.78e-17, -1.67e-01, -1.67e-01, -8.33e-01, -8.33e-01, +3.33e-01, +3.33e-01, -3.33e-01, -3.33e-01, ],
    [-3.33e-01, -5.00e-01, -6.67e-01, -8.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, -6.67e-01, -8.33e-01, -1.00e+00, -1.17e+00, -1.67e-01, -3.33e-01, -5.00e-01, -6.67e-01, +6.67e-01, +5.00e-01, +3.33e-01, +1.67e-01, +1.17e+00, +1.00e+00, +8.33e-01, +6.67e-01, +3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, +8.33e-01, +6.67e-01, +5.00e-01, +3.33e-01, -3.33e-01, -5.00e-01, -6.67e-01, -8.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, -6.67e-01, -8.33e-01, -1.00e+00, -1.17e+00, -1.67e-01, -3.33e-01, -5.00e-01, -6.67e-01, +6.67e-01, +5.00e-01, +3.33e-01, +1.67e-01, +1.17e+00, +1.00e+00, +8.33e-01, +6.67e-01, +3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, +8.33e-01, +6.67e-01, +5.00e-01, +3.33e-01, ],
    [+0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, +0.00e+00, -1.67e-01, +1.67e-01, +0.00e+00, +3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, +5.00e-01, +3.33e-01, +6.67e-01, +5.00e-01, +8.33e-01, +6.67e-01, +1.00e+00, +8.33e-01, +5.00e-01, +3.33e-01, +6.67e-01, +5.00e-01, +8.33e-01, +6.67e-01, +1.00e+00, +8.33e-01, -1.67e-01, -3.33e-01, +0.00e+00, -1.67e-01, +1.67e-01, +2.78e-17, +3.33e-01, +1.67e-01, -1.67e-01, -3.33e-01, +0.00e+00, -1.67e-01, +1.67e-01, +2.78e-17, +3.33e-01, +1.67e-01, +3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, +6.67e-01, +5.00e-01, +8.33e-01, +6.67e-01, +3.33e-01, +1.67e-01, +5.00e-01, +3.33e-01, +6.67e-01, +5.00e-01, +8.33e-01, +6.67e-01, ],
    [+0.00e+00, -1.67e-01, -3.33e-01, -5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, -5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, +5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, +8.33e-01, +6.67e-01, +5.00e-01, +3.33e-01, +5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, +8.33e-01, +6.67e-01, +5.00e-01, +3.33e-01, +0.00e+00, -1.67e-01, -3.33e-01, -5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, +0.00e+00, -1.67e-01, -3.33e-01, -5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, -1.67e-01, +5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, +8.33e-01, +6.67e-01, +5.00e-01, +3.33e-01, +5.00e-01, +3.33e-01, +1.67e-01, +0.00e+00, +8.33e-01, +6.67e-01, +5.00e-01, +3.33e-01, ],
    [+0.00e+00, -3.33e-01, +0.00e+00, -3.33e-01, +3.33e-01, +0.00e+00, +3.33e-01, +0.00e+00, +0.00e+00, -3.33e-01, +0.00e+00, -3.33e-01, +3.33e-01, +0.00e+00, +3.33e-01, +0.00e+00, +1.00e+00, +6.67e-01, +1.00e+00, +6.67e-01, +1.33e+00, +1.00e+00, +1.33e+00, +1.00e+00, +1.00e+00, +6.67e-01, +1.00e+00, +6.67e-01, +1.33e+00, +1.00e+00, +1.33e+00, +1.00e+00, +0.00e+00, -3.33e-01, +0.00e+00, -3.33e-01, +3.33e-01, +0.00e+00, +3.33e-01, +0.00e+00, +0.00e+00, -3.33e-01, +0.00e+00, -3.33e-01, +3.33e-01, +0.00e+00, +3.33e-01, +0.00e+00, +1.00e+00, +6.67e-01, +1.00e+00, +6.67e-01, +1.33e+00, +1.00e+00, +1.33e+00, +1.00e+00, +1.00e+00, +6.67e-01, +1.00e+00, +6.67e-01, +1.33e+00, +1.00e+00, +1.33e+00, +1.00e+00, ],
    ]
    
    A = -np.asarray(A)
    
    n=27
    m=64
    
    start = time.time()
    if alg == 'rmp':
        print('running RM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y=regret_matching_plus(T,A,n,m,q,R0)    
    elif alg == 'alt_rmp':
        print('running alt-RM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y=regret_matching_alt_plus(T,A,n,m,q,R0) 
    elif alg == 'prmp':
        print('running PRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y=optimistic_regret_matching_plus(T,A,n,m,q,R0)   
    elif alg == 'alt_prmp':
        print('running alt-PRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_regret_matching_plus_alt(T,A,n,m,q,R0)  
    elif alg == 'crmp':
        print('running ExRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = conceptual_regret_matching_plus_chopped(T,A,n,m,q,k,R0,eta)   
    elif alg == 'crmp_rs':
        print('running RS-ExRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = conceptual_regret_matching_plus_chopped_restarting(T,A,n,m,q,k,R0,eta)   
    elif alg == 'smooth_prmp':
        print('running smooth-PRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_regret_matching_plus_chopped(T,A,n,m,q,R0,eta)
    elif alg == 'smooth_prmp_rs':
        print('running RS-smooth-PRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_regret_matching_plus_chopped_restarting(T,A,n,m,q,R0,eta)
    elif alg == 'alt_smooth_prmp':
        print('running alt-smooth-PRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_regret_matching_plus_alt_chopped(T,A,n,m,q,R0,eta)
    elif alg == 'extragradient_alg':
        print('running extragradient_algorithm...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = extragradient_algorithm(T,A,n,m,eta)
    elif alg == 'optimistic_omd':
        print('running optimistic_omd...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_omd(T,A,n,m,eta)
     
        
    end = time.time()
    print('Running time:', np.around(end-start,2))
    
    np.save(f'numerical_results/{alg}_duality_gap_{T}_kuhn_eta_{eta}.npy',result_duality_gap)        
    np.save(f'numerical_results/{alg}_results_x_{T}_kuhn_eta_{eta}.npy',results_x)        
    np.save(f'numerical_results/{alg}_results_y_{T}_kuhn_eta_{eta}.npy',results_y)        
    np.save(f'numerical_results/{alg}_regret_x_{T}_kuhn_eta_{eta}.npy',regret_x)        
    np.save(f'numerical_results/{alg}_regret_y_{T}_kuhn_eta_{eta}.npy',regret_y) 
    
    x_axis = np.asarray(range(T-1))
    indices = np.logspace(0.0, np.log10(len(x_axis)), num=nb_points)
    indices=indices.astype(int)
    indices[-1]=indices[-1]-2
    
    ######### Plot the duality gap

    fig, ax = plt.subplots()
    
    l1 =  ax.plot(x_axis[indices], result_duality_gap[indices], color='red', marker='*', linestyle=':',linewidth=2, markersize=10,label = 'Duality gap')

    plt.xscale('log')
    plt.yscale('log')
    
    ax.set_xlabel('Number of iterations',fontsize=14)
    ax.set_ylabel('Duality gap',fontsize=14)
    ax.tick_params(axis='both', which='major', labelsize=14)

    ax.grid()
    
    fig.legend(ncol=2,loc="upper center",bbox_to_anchor=(0.5,1.2,0,0),fontsize=15)
            
    fig.savefig(f"figures/duality_gap_T_{T}_{alg}_kuhn_eta_{eta}.pdf", bbox_inches='tight')
    plt.show()
   
if run_goofspiel:
    
    # Load the matrix of the game
    A = np.load('U_goofspiel.npy')
    
    m=len(A[0])
    n=len(A)
    
    start = time.time()
    if alg == 'rmp':
        print('running RM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y=regret_matching_plus(T,A,n,m,q,R0)    
    elif alg == 'alt_rmp':
        print('running alt-RM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y=regret_matching_alt_plus(T,A,n,m,q,R0) 
    elif alg == 'prmp':
        print('running PRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y=optimistic_regret_matching_plus(T,A,n,m,q,R0)   
    elif alg == 'alt_prmp':
        print('running alt-PRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_regret_matching_plus_alt(T,A,n,m,q,R0)  
    elif alg == 'crmp':
        print('running ExRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = conceptual_regret_matching_plus_chopped(T,A,n,m,q,k,R0,eta)   
    elif alg == 'crmp_rs':
        print('running RS-ExRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = conceptual_regret_matching_plus_chopped_restarting(T,A,n,m,q,k,R0,eta)   
    elif alg == 'smooth_prmp':
        print('running smooth-PRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_regret_matching_plus_chopped(T,A,n,m,q,R0,eta)
    elif alg == 'smooth_prmp_rs':
        print('running RS-smooth-PRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_regret_matching_plus_chopped_restarting(T,A,n,m,q,R0,eta)
    elif alg == 'alt_smooth_prmp':
        print('running alt-smooth-PRM+...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_regret_matching_plus_alt_chopped(T,A,n,m,q,R0,eta)
    elif alg == 'extragradient_alg':
        print('running extragradient_algorithm...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = extragradient_algorithm(T,A,n,m,eta)
    elif alg == 'optimistic_omd':
        print('running optimistic_omd...')
        result_duality_gap, results_x, results_y, regret_x, regret_y = optimistic_omd(T,A,n,m,eta)
     
    end = time.time()
    print('Running time:', np.around(end-start,2))
    
    np.save(f'numerical_results/{alg}_duality_gap_{T}_goofspiel_eta_{eta}.npy',result_duality_gap)        
    np.save(f'numerical_results/{alg}_results_x_{T}_goofspiel_eta_{eta}.npy',results_x)        
    np.save(f'numerical_results/{alg}_results_y_{T}_goofspiel_eta_{eta}.npy',results_y)        
    np.save(f'numerical_results/{alg}_regret_x_{T}_goofspiel_eta_{eta}.npy',regret_x)        
    np.save(f'numerical_results/{alg}_regret_y_{T}_goofspiel_eta_{eta}.npy',regret_y) 
    
    x_axis = np.asarray(range(T-1))
    indices = np.logspace(0.0, np.log10(len(x_axis)), num=nb_points)
    indices=indices.astype(int)
    indices[-1]=indices[-1]-2
    
    ######### Plot the duality gap

    fig, ax = plt.subplots()
    
    l1 =  ax.plot(x_axis[indices], result_duality_gap[indices], color='red', marker='*', linestyle=':',linewidth=2, markersize=10,label = 'Duality gap')

    plt.xscale('log')
    plt.yscale('log')
    
    ax.set_xlabel('Number of iterations',fontsize=14)
    ax.set_ylabel('Duality gap',fontsize=14)
    ax.tick_params(axis='both', which='major', labelsize=14)

    ax.grid()
    
    fig.legend(ncol=2,loc="upper center",bbox_to_anchor=(0.5,1.2,0,0),fontsize=15)
            
    fig.savefig(f"figures/duality_gap_T_{T}_{alg}_goofspiel_eta_{eta}.pdf", bbox_inches='tight')
    plt.show()



    
    